<?php
namespace App\Http\Controllers\MobileApp;

use App\Http\Controllers\Controller;
use App\Models\ModelSlider;
use App\Models\Category;
use App\Models\SparePart;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;

class HomeController extends Controller
{
    /**
     * Constructor to set locale
     */
    public function __construct(Request $request)
    {
        $locale = $request->header('Accept-Language', 'ar');
        App::setLocale($locale);
    }

    /**
     * Get home page data (Slider, Categories, Latest Spare Parts)
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        $locale = App::getLocale(); // Get the current locale (e.g., 'en' or 'ar')

        // Get sliders
        $sliders = ModelSlider::all()->map(function ($slider) use ($locale) {
            return [
                'id' => $slider->id,
                'image_url' => $slider->image_url,
                'title' => json_decode($slider->title)->{$locale} ?? '', // Dynamically select locale
            ];
        });

        // Get categories
        $categories = Category::all()->map(function ($category) use ($locale) {
            return [
                'id' => $category->id,
                'name' => json_decode($category->name)->{$locale} ?? '', // Dynamically select locale
                'image' => $category->image,
                'is_active' => $category->is_active,
            ];
        });

        // Get latest 5 spare parts
        $latestSpareParts = SparePart::with('category')
            ->latest()
            ->take(5)
            ->get()
            ->map(function ($sparePart) use ($locale) {
                return [
                    'id' => $sparePart->id,
                    'name' => json_decode($sparePart->name)->{$locale} ?? '', // Dynamically select locale
                    'price' => $sparePart->price,
                    'stock_quantity' => $sparePart->stock_quantity,
                    'image' => $sparePart->image,
                    'discount' => null,
                    // Uncomment if you want to include category name
                    // 'category_name' => $sparePart->category ? json_decode($sparePart->category->name)->{$locale} ?? '' : '',
                ];
            });

        return response()->json([
            'status' => 'success',
            'message' => trans('messages.home_retrieved'),
            'data' => [
                'sliders' => $sliders,
                'categories' => $categories,
                'latest_spare_parts' => $latestSpareParts,
            ],
        ], 200);
    }
}